<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Providers\RouteServiceProvider;
use Cache;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    public function login(Request $request)
    {
        $this->validateLogin($request);

        $user = \App\Models\User::where('email', $request->email)->first();
        if (!$user) {
            throw ValidationException::withMessages([
                'email' => ['Estas credenciales no coinciden con nuestros registros.'],
            ]);
        }

        $user_role = $user->roles[0];

        if ($user && $user_role->reference === 'admin') {
            $cachedValue = Cache::get('user-ping-' . $user->id);

            if ($cachedValue) {
                throw ValidationException::withMessages([
                    'email' => ['Ya hay una sesión activa como administrador.'],
                ]);
            }
        }

        if (!$user_role->is_active) {
            throw ValidationException::withMessages([
                'email' => ['No puede acceder al sistema, su rol se encuentra desactivado.'],
            ]);
        }

        if ($this->attemptLogin($request)) {
            $user->update(['session_id' => session()->getId()]);
            Cache::put('user-ping-' . $user->id, now(), 600);

            return $this->sendLoginResponse($request);
        }


        return $this->sendFailedLoginResponse($request);
    }

    public function logout(Request $request)
    {
        $user = Auth::user();
        if ($user) {
            Cache::forget('user-ping-' . $user->id);
            $user->update(['session_id' => null]);
        }

        $this->guard()->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }

    protected function authenticated(Request $request, $user)
    {
        $role = $user->roles()->first();
        if ($role && $role->reference === 'client') {
            return redirect('/catalog');

        } else {
            return redirect('/home');
        }
    }
}
